import 'dart:html';

import 'package:photogram_admin_cp/ui/components/layout.dart';
import 'package:rad/rad.dart';
import 'package:rad/widgets_html.dart';

import 'package:photogram_admin_cp/import/data.dart';
import 'package:photogram_admin_cp/import/core.dart';

class UserField {
  String key;
  FieldType type;
  String leading;
  String? description;
  String? placeholder;

  UserField({
    required this.key,
    required this.type,
    required this.leading,
    this.description,
    this.placeholder,
  });
}

class UserEditorPage extends StatefulWidget {
  final String? heading;
  final String? description;

  final List<UserField> fields;

  const UserEditorPage({
    Key? key,
    required this.fields,
    this.heading,
    this.description,
  }) : super(key: key);

  @override
  _UserEditorPageState createState() => _UserEditorPageState();
}

class _UserEditorPageState extends State<UserEditorPage> with AppActiveContentMixin {
  int _userToEditId = 0;

  final _fields = <UserField>[];

  UserModel _userModel = UserModel.none();

  var _userDataMap = <String, dynamic>{};

  var _isUnsaved = false;
  var _isSaveInProgress = false;
  var _saveError = '';

  @override
  onLoadEvent() {
    _fields.addAll(widget.fields);
  }

  @override
  void didChangeDependencies() {
    var userId = AppUtils.intVal(
      Navigator.of(context, byKey: const Key('home-screen-navigator')).getValue('user'),
    );

    _isUnsaved = false;
    _isSaveInProgress = false;
    _saveError = '';

    _userModel = activeContent.read<UserModel>(userId)!;
    _userDataMap = _userModel.toJson();

    if (userId != _userToEditId) {
      if (0 != _userToEditId) {
        _userToEditId = userId;

        Navigator.of(context).open(name: 'display');
      } else {
        _userToEditId = userId;
      }
    }
  }

  @override
  build(context) {
    return PageRow(
      children: [
        Division(
          className: 'col-lg-6',
          children: [
            Division(
              className: 'content',
              children: [
                // page heading and description

                if (null != widget.heading)
                  Heading1(className: 'content-title font-weight-medium', innerText: widget.heading),

                if (null != widget.description)
                  Division(className: 'text-muted font-size-12', innerText: widget.description),

                // fields

                for (var item in _fields) _getEditingField(item),

                // break lines to prevent overlay

                const LineBreak(),
                const LineBreak(),
                const LineBreak(),
                const LineBreak(),
                const LineBreak(),

                // unsaved detector

                if (_isUnsaved)
                  Division(
                    style: 'position: fixed; bottom: 80px; max-width: 665px; width:100%;',
                    children: [
                      Division(
                        className: 'alert animate__animated animate__slideInUp'
                            ' ${_saveError.isNotEmpty ? 'alert-danger' : 'alert-secondary'} ',
                        children: [
                          Division(
                            className: 'clearfix',
                            children: [
                              Division(
                                className: 'float-left d-inline-block font-size-14 font-weight-medium',
                                innerText: _saveError.isNotEmpty ? 'Whoops' : 'Unsaved Changes',
                              ),
                              Division(
                                className: 'float-right d-inline-block ',
                                child: Button(
                                  className: 'btn alt-dm',
                                  onClick: _saveUserData,
                                  disabled: _isSaveInProgress,
                                  innerText:
                                      _isSaveInProgress ? 'Saving...' : (_saveError.isNotEmpty ? 'Retry' : 'Save'),
                                ),
                              ),
                            ],
                          ),
                          Division(
                            className: 'font-size-12 text-muted my-5',
                            innerText: _saveError.isNotEmpty ? _saveError : "Careful - You've unsaved changes",
                          ),
                        ],
                      ),
                    ],
                  ),
              ],
            ),
          ],
        ),
      ],
    );
  }

  Widget _getEditingField(UserField field) {
    switch (field.type) {
      case FieldType.switchable:
        return Division(
          children: [
            const HorizontalRule(className: 'my-20'),
            Division(
              className: 'clearfix',
              children: [
                Division(
                  className: 'float-left d-inline-block font-size-14 font-weight-medium',
                  innerText: field.leading,
                ),
                Division(
                  className: 'float-right d-inline-block ',
                  child: Division(
                    className: 'custom-switch',
                    children: [
                      InputCheckBox(
                          id: UserTable.tableName + field.key,
                          checked: '1' == _userDataMap[field.key],
                          onChange: (event) {
                            setState(() {
                              _refreshUnsaveState();
                            });
                          }),
                      Label(forAttribute: field.key),
                    ],
                  ),
                ),
              ],
            ),
            Division(
              className: 'font-size-12 my-10 text-muted',
              innerText: field.description,
            ),
          ],
        );

      case FieldType.text:
        return Division(
          children: [
            const HorizontalRule(className: 'my-20'),
            Division(
              children: [
                Division(
                  className: 'font-size-14 font-weight-medium',
                  innerText: field.leading,
                ),
                InputText(
                  id: UserTable.tableName + field.key,
                  className: 'form-control my-10 px-10 py-15',
                  placeholder: field.placeholder,
                  value: _userDataMap[field.key],
                  onChange: (event) {
                    setState(() {
                      _refreshUnsaveState();
                    });
                  },
                ),
              ],
            ),
            Division(
              className: 'font-size-12 my-10 text-muted',
              innerText: field.description,
            ),
          ],
        );

      case FieldType.select:
        return const Division(innerText: 'Not implemented/required for user fields');

      case FieldType.textarea:
        return Division(
          children: [
            const HorizontalRule(className: 'my-20'),
            Division(
              children: [
                Division(
                  className: 'font-size-14 font-weight-medium',
                  innerText: field.leading,
                ),
                TextArea(
                    id: UserTable.tableName + field.key,
                    className: 'form-control my-10 px-10 py-15',
                    placeholder: field.placeholder,
                    innerText: _userDataMap[field.key],
                    onChange: (event) {
                      setState(() {
                        _refreshUnsaveState();
                      });
                    }),
              ],
            ),
            Division(
              className: 'font-size-12 my-10 text-muted',
              innerText: field.description,
            ),
          ],
        );
    }
  }

  String? _getFieldValue(UserField field) {
    var element = document.getElementById(UserTable.tableName + field.key);

    switch (field.type) {
      case FieldType.textarea:
        element as TextAreaElement;

        return element.value;

      default:
        element as InputElement;
        return element.value;
    }
  }

  void _refreshUnsaveState() {
    _userDataMap = _userModel.toJson();

    var isAnyValueChanged = false;

    for (var field in _fields) {
      var originalValue = _userDataMap[field.key];
      var currentValue = _getFieldValue(field);

      if (currentValue != originalValue) {
        isAnyValueChanged = true;

        break;
      }
    }

    if (isAnyValueChanged && !_isUnsaved) {
      _isUnsaved = true;
      _saveError = '';

      return;
    }

    if (!isAnyValueChanged && _isUnsaved) {
      _isUnsaved = false;
      _saveError = '';

      return;
    }
  }

  void _saveUserData(EmittedEvent event) async {
    if (_isSaveInProgress) return;

    setState(() {
      _saveError = '';
      _isSaveInProgress = true;
    });

    try {
      var apiRepo = activeContent.apiRepository;

      var userUpdateData = {UserTable.id: '$_userToEditId'};

      for (var field in _fields) {
        var value = _getFieldValue(field);

        if (null != value) {
          userUpdateData[field.key] = value;
        }
      }

      var responseModel = await apiRepo.preparedRequest(
        requestType: REQ_TYPE_ADMIN_USER_SAVE_SINGLE,
        requestData: {
          UserTable.tableName: userUpdateData,
        },
      );

      switch (responseModel.message) {
        case SUCCESS_MSG:
          break;

        // username

        case D_ERROR_USER_USERNAME_MAX_LEN_MSG:
          _saveError = 'Username is bit long.';
          break;

        case D_ERROR_USER_USERNAME_MIN_LEN_MSG:
          _saveError = 'Username is bit small.';
          break;

        case D_ERROR_USER_USERNAME_NOT_AVAILABLE_MSG:
          _saveError = 'Username is already in use.';
          break;

        case D_ERROR_USER_USERNAME_STARTS_WITH_ALHPABET_MSG:
          _saveError = 'Username must starts with a alphabet characeter.';
          break;

        case D_ERROR_USER_USERNAME_ALLOWED_CHARACTERS_ONLY_MSG:
          _saveError = 'Username contains special characters that are not allowed.';
          break;

        // email

        case D_ERROR_USER_EMAIL_MAX_LEN_MSG:
          _saveError = 'Email address is bit large.';
          break;

        case D_ERROR_USER_EMAIL_MIN_LEN_MSG:
          _saveError = 'Email address is bit small.';
          break;

        case D_ERROR_USER_EMAIL_INVALID_FORMAT_MSG:
          _saveError = 'Email address is invalid.';
          break;

        case D_ERROR_USER_EMAIL_NOT_AVAILABLE_MSG:
          _saveError = 'Email address is already in use.';
          break;

        // display name

        case D_ERROR_USER_DISPLAY_NAME_MAX_LEN_MSG:
          _saveError = 'Display name is bit large.';
          break;

        case D_ERROR_USER_DISPLAY_NAME_MIN_LEN_MSG:
          _saveError = 'Display name is bit small.';
          break;

        // display web

        case D_ERROR_USER_DISPLAY_WEB_MAX_LEN_MSG:
          _saveError = 'Display web is bit large.';
          break;

        case D_ERROR_USER_DISPLAY_WEB_MIN_LEN_MSG:
          _saveError = 'Display web is bit small.';
          break;

        case D_ERROR_USER_DISPLAY_WEB_INVALID_FORMAT_MSG:
          _saveError = 'Web address is not valid.';
          break;

        // display bio

        case D_ERROR_USER_DISPLAY_BIO_MAX_LEN_MSG:
          _saveError = 'Display bio is bit large.';
          break;

        case D_ERROR_USER_DISPLAY_BIO_MIN_LEN_MSG:
          _saveError = 'Display bio is bit small.';
          break;

        default:
          _saveError = 'Something went wrong';
          break;
      }

      if (_saveError.isNotEmpty) {
        return setState(() {
          _isSaveInProgress = false;
        });
      }

      activeContent.handleResponse(responseModel);
    } catch (e) {
      AppLogger.exception(e);
    }

    setState(() {
      _refreshUnsaveState();

      _isSaveInProgress = false;
    });
  }
}
